/*****************************************************************************/
/*                                                                           */
/*       --- CAEN Engineering Srl - Computing Systems Division ---           */
/*                                                                           */
/*   A303drv.c                                                               */
/*                                                                           */
/*   Linux device driver for CAEN Mod. A303 or A303A                         */
/*   Source file of the driver.                                              */
/*   The driver works with A303 and A303A, because they are software         */
/*   compatible.                                                             */
/*                                                                           */
/*   Created                                                     - May 1999  */
/*   Corrected Bug in IOctl                                      - Nov 1999  */
/*   Corrected Bug in A303Write                                  - Jan 2000  */
/*   Eliminated the I/O - Memory option: we work only in I/O now - Jan 2000  */
/*   Porting to the 2.2 kernel varsion                           - Jun 2000  */
/*                                                                           */
/*****************************************************************************/
#include <linux/fs.h>      /* To register new driver                  */
#include <linux/errno.h>   /* Error Codes                             */
#include <linux/sched.h>   /* To sleep and wake up                    */
#include <linux/mm.h>      /* To verify memory between transfers      */
#include <linux/poll.h>    /* For the poll system call implemantation */

#ifdef MODULE
#include <linux/module.h>
#endif

#include <asm/segment.h>
#include <asm/io.h>
#include "a303drv.h"

typedef unsigned char uchar;

/*
  A303 Registers offsets
*/
#define   A303_FIFO                (0)
#define   A303_REG                 (1)
#define   A303_INTR                (2)
#define   A303_RESET               (3)

/* Status Register Masks */
#define   NOINTR                   0x26
#define   RXFEM                       1
#define   IDLE      (unsigned char)0xee
#define   TXEFF                    0x20
#define   RXEFF                    0x04

#define   TIME_OUT                 100000
#define   TIME_OUT_TX              100000

#define   TUTTOK                    0
#define   E_NO_A303                -2

#define   MAX_LENGTH_FIFO          4096

#define   A303_IRQ                  (9)

static struct state
{
int      major;
int      irq;
struct   wait_queue *proc_list;
int      timeout;
int      rx_ready;
int      buff_count;
unsigned offset;
uchar    buff[MAX_LENGTH_FIFO];
} A303 = { 0, A303_IRQ, NULL, 100, 0, 0, 0x300, };


/******************************************************************************/
/*                                                                            */
/* A303RESET                                                                  */
/*                                                                            */
/******************************************************************************/
static int A303Reset(void)
{
 int i = 0;
 
outb(0, A303.offset + A303_RESET);
do
 i++;
while( inb(A303.offset + A303_REG) != IDLE && i != TIME_OUT );
return ( (i == TIME_OUT) ? E_NO_A303 : TUTTOK );
}

/******************************************************************************/
/*                                                                            */
/* A303INTERRUPT                                                              */
/* This is the interrupt handler.                                             */
/*  A303 Module rises an interrupt when one of the following events occurs:   */
/*   1) data packet transmission completed                                    */
/*   2) end of data packet reading                                            */
/*   3) data packet received                                                  */
/*  We decide to permit only the occurring of the last one, while the first   */
/*  one will be inhibited during the transmission routine and the second one  */
/*  will be reset here, at the end of the reading.                            */
/*  Remember that A303 and A303A use Interrupt 9; in particular:              */
/*  On A303A if SW7 On -> Interrupt enabled                                   */
/*  On A303  if SW2 On -> Interrupt enabled                                   */
/*                                                                            */
/******************************************************************************/
static void A303Interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
uchar A303_stat;

A303_stat = inb(A303.offset+A303_REG);

if( !(A303_stat & RXEFF) )                             /* A packet has come */
   {
    while( inb(A303.offset+A303_REG) & RXFEM )              /* Data available */
      {
       A303.buff[A303.buff_count] = inb(A303.offset);
       A303.buff_count++;
      }
    outb(0, A303.offset + A303_RESET);        /* Reset Interrupt 3)       */
    A303.rx_ready = 1;                         /* Flag for the read routine */
    wake_up_interruptible(&A303.proc_list);
   }
}

/*****************************************************************************/
/*                                                                           */
/* A303SELECT                                                                */
/* The select function                                                       */
/*                                                                           */
/*****************************************************************************/
static unsigned int A303Select(struct file *file, poll_table *wait)
{
  unsigned int mask = 0;

  if( A303.rx_ready )
     return POLLIN | POLLRDNORM;
  poll_wait(file, &A303.proc_list, wait);
  if( A303.rx_ready )
     return POLLIN | POLLRDNORM;
  return 0;
}

/*****************************************************************************/
/*                                                                           */
/* A303READ                                                                  */
/* The read function.                                                        */
/* Waits A303.timeout "jiffies" before deciding that no data are available.  */
/* If data are available, it transfers them in buf for the user.             */
/*                                                                           */
/*****************************************************************************/
static ssize_t A303Read(struct file *file, char *buf, size_t count, loff_t *ppos)
{
int ret;

cli();
while( !A303.rx_ready ) 
 {
    interruptible_sleep_on_timeout(&A303.proc_list, A303.timeout);
    if( signal_pending(current) )
       return -EINTR;
    if( !A303.rx_ready )
       return -ETIMEDOUT;
 }

A303.rx_ready = 0;
if( ret = verify_area(VERIFY_WRITE, (void *)buf, A303.buff_count) )
   return ret;

if( copy_to_user(buf, &A303.buff, A303.buff_count) > 0)
  return -EFAULT;

sti();
ret = A303.buff_count;
A303.buff_count = 0;
return ret;
}

/*****************************************************************************/
/*                                                                           */
/* A303WRITE                                                                 */
/* The write function.                                                       */
/* Reset A303, send data, wait end of transmission                           */
/*                                                                           */
/*****************************************************************************/
static ssize_t A303Write(struct file *file, const char *buf, size_t count, loff_t *ppos)
{
int rc, i, j;

if( rc = verify_area(VERIFY_READ, buf, count) )
   return rc;
cli();
if( copy_from_user(&A303.buff, buf, count) > 0)
  return -EFAULT;

/* Reset A303 Module */
if( A303Reset() != 0 )
   {
    printk("There are problems in Caenet Board\n");
    sti();
    return -1;
   }
   
A303.rx_ready = 0;      // Added Jan 2000  

/* Fill Transmit buffer */
for( i = 0; i < count; i++ )
    outb(A303.buff[i], A303.offset+A303_FIFO);

/* Send data */
outb(0, A303.offset+A303_REG); 

/* Wait until transmission is done and, at the end, reset Interrupt 1) */
i = 0;
while( (inb(A303.offset+A303_REG) & TXEFF) && i<TIME_OUT_TX ) i++;
j = inb(A303.offset+A303_INTR);         // Jan 2000 - Dummy read
sti();

return((i == TIME_OUT_TX) ? -ETIMEDOUT : count);
}

/******************************************************************************/
/*                                                                            */
/* A303IOCTL                                                                  */
/* The ioctl function.                                                        */
/*                                                                            */
/******************************************************************************/
static int A303Ioctl(struct inode *node, struct file *file, unsigned int cmd, unsigned long arg)
{
int i, val = 0;

i = 0;                                               // Introduced Nov 1999

switch( cmd )
  {
/* It is the time which the read function waits before than signalling  */
/* that the addressed slave module is not present.                      */
/* The following relation holds: timeout (in sec.) = time_out / 100     */
   case TIMEOUT_A303:
     A303.timeout = arg;
     break;

/* Sets the A303 module base address in PC memory map.                  */
/* On A303A the address must be set using SW1 ... SW4 (see user manual) */
/* On A303  the address must be set using SW3 SW4 SW5 (see user manual) */
/* Remember that this base address is referred to memory or I/O         */
/* depending on setting of SW7 (A303A) or SW1 (A303)                    */
   case ADDRESS_A303:
     A303.offset = arg;
     break;

/* Executes an hardware reset of A303 module                            */
   case RESET_A303:
     outb(0, A303.offset + A303_RESET);
     do
      i++;
     while( inb(A303.offset + A303_REG) != IDLE && i != TIME_OUT );
     val = ( (i == TIME_OUT) ? -ETIMEDOUT : TUTTOK );
     break;

   default:
     val = -EINVAL;
     break;
  }
return val;
}

/******************************************************************************/
/*                                                                            */
/* A303OPEN                                                                   */
/* The open function.                                                         */
/*                                                                            */
/******************************************************************************/
static int A303Open(struct inode *node, struct file *file)
{
int ret;

cli();

MOD_INC_USE_COUNT;

if( A303.irq )
   {
    ret = request_irq(A303.irq, A303Interrupt, SA_INTERRUPT, "A303", NULL);
    sti();
    if( ret )
       {
        printk("IRQ %d not available\n",A303.irq);
        MOD_DEC_USE_COUNT;
        return ret;
       }
   }
return 0;
}

/******************************************************************************/
/*                                                                            */
/* A303RELEASE                                                                */
/*                                                                            */
/******************************************************************************/
static int A303Release(struct inode * inode, struct file * file)
{
if( A303.irq ) 
   free_irq(A303.irq,NULL);
MOD_DEC_USE_COUNT;
}

static struct file_operations A303_fops = {
  NULL,             /* lseek              */
  A303Read,         /* read               */
  A303Write,        /* write              */
  NULL,             /* readdir            */
  A303Select,       /* poll               */
  A303Ioctl,        /* ioctl              */
  NULL,             /* mmap               */
  A303Open,         /* open               */
  NULL,             /* flush              */
  A303Release,      /* release            */
  NULL,             /* fsync              */
  NULL,             /* fasync             */
  NULL,             /* check_media_change */
  NULL,             /* revalidate         */
};

#ifndef MODULE
/* Static link to kernel */
long init_A303(void)
{
if( A303.major = register_chrdev(A303_MAJOR, "A303", &A303_fops) )
   printk("unable to get major for A303 device\n");
return 0;
}
#else
int init_module(void)
{
    int result = register_chrdev(A303.major, "A303", &A303_fops);
    if (result < 0) {
        printk(KERN_INFO "A303: can't get major number\n");
        return result;
    }
    if (A303.major == 0) A303.major = result; /* dynamic */
    return 0;
}

/******************************************************************************/
/*                                                                            */
/* CLEANUP_MODULE                                                             */
/*                                                                            */
/******************************************************************************/
void cleanup_module(void)
{
 unregister_chrdev(A303.major, "A303");
}
#endif
